package com.androidbook.simplendk;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import javax.microedition.khronos.egl.EGLConfig;
import javax.microedition.khronos.opengles.GL10;
import android.app.Activity;
import android.content.Context;
import android.opengl.GLES20;
import android.opengl.GLSurfaceView;
import android.os.Bundle;
import android.util.Log;


public class NativeOpenGL2Activity extends Activity{
    private static final String DEBUG_TAG = "NativeOpenGL2Activity";
    CustomGL2SurfaceView mAndroidSurface = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mAndroidSurface = new CustomGL2SurfaceView(this);
        setContentView(mAndroidSurface);
    }

    @Override
    protected void onPause() {
        super.onPause();
        if (mAndroidSurface != null) {
            mAndroidSurface.onPause();
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        if (mAndroidSurface != null) {
            mAndroidSurface.onResume();
        }
    }

    private class CustomGL2SurfaceView extends GLSurfaceView {
        final CustomRenderer renderer;

        public CustomGL2SurfaceView(Context context) {
            super(context);
            // Pobranie kontekstu OpenGL ES 2.0.
            setEGLContextClientVersion(2);
            renderer = new CustomRenderer();
            setRenderer(renderer);
        }
    }

    private class CustomRenderer implements GLSurfaceView.Renderer {
        private boolean initialized = false;

        @Override
        public void onDrawFrame(GL10 unused) {
            if (!initialized) {
                return;
            }
            drawFrame(shaderProgram);
        }

        @Override
        public void onSurfaceChanged(GL10 unused, int width, int height) {
            Log.v(DEBUG_TAG, "onSurfaceChanged");
            GLES20.glViewport(0, 0, width, height);
            GLES20.glClearColor(0.5f, 0.5f, 0.5f, 1);
        }

        @Override
        public void onSurfaceCreated(GL10 unused, EGLConfig unused2) {
            try {
                initShaderProgram(R.raw.simple_vertex, R.raw.simple_fragment);
                initialized = true;
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "Nie udało się zainicjować GL");
            }
        }

        private int shaderProgram = 0;

        private void initShaderProgram(int vertexId, int fragmentId)
            throws Exception {
            int vertexShader =
                loadAndCompileShader(GLES20.GL_VERTEX_SHADER, vertexId);
            int fragmentShader =
                loadAndCompileShader(GLES20.GL_FRAGMENT_SHADER, fragmentId);
            shaderProgram = GLES20.glCreateProgram();
            if (shaderProgram == 0) {
                throw new Exception("Nie udało się utworzyć programu cieniującego.");
            }
            // dodanie programu cieniującego
            GLES20.glAttachShader(shaderProgram, vertexShader);
            GLES20.glAttachShader(shaderProgram, fragmentShader);
            // powiązanie atrybutu z programem cieniowania wierzchołków
            GLES20.glBindAttribLocation(shaderProgram, 0, "vPosition");
            // konsolidacja programu cieniującego
            GLES20.glLinkProgram(shaderProgram);
            // sprawdzenie statusu konsolidacji
            int[] linkerStatus = new int[1];
            GLES20.glGetProgramiv(shaderProgram, GLES20.GL_LINK_STATUS,
                linkerStatus, 0);
            if (GLES20.GL_TRUE != linkerStatus[0]) {
                Log.e(DEBUG_TAG, "Błąd konsolidacji: "
                    + GLES20.glGetProgramInfoLog(shaderProgram));
                GLES20.glDeleteProgram(shaderProgram);
                throw new Exception("Nieudana konsolidacja programu cieniującego.");
            }
            GLES20.glClearColor(0.5f, 0.5f, 0.5f, 1);
        }

        private int loadAndCompileShader(int shaderType, int shaderId)
            throws Exception {
            InputStream inputStream =
                NativeOpenGL2Activity.this.getResources().openRawResource(shaderId);
            String shaderCode = inputStreamToString(inputStream);
            int shader = GLES20.glCreateShader(shaderType);
            if (shader == 0) {
                throw new Exception("Nie można utworzyć programu cieniującego.");
            }
            // Przekazanie kodu do GL.
            GLES20.glShaderSource(shader, shaderCode);
            // Kompilacja
            GLES20.glCompileShader(shader);
            // Pobranie statusu kompilacji.
            int[] status = new int[1];
            GLES20.glGetShaderiv(shader, GLES20.GL_COMPILE_STATUS, status, 0);
            if (status[0] == 0) {
                // niepowodzenie
                Log.e(DEBUG_TAG, "Błąd kompilacji: "
                    + GLES20.glGetShaderInfoLog(shader));
                GLES20.glDeleteShader(shader);
                throw new Exception("Nieudana kompilacja programu cieniującego.");
            }
            return shader;
        }

        public String inputStreamToString(InputStream is) throws IOException {
            StringBuffer sBuffer = new StringBuffer();
            DataInputStream dataIO = new DataInputStream(is);
            String strLine = null;
            while ((strLine = dataIO.readLine()) != null) {
                sBuffer.append(strLine + "\n");
            }
            dataIO.close();
            is.close();
            return sBuffer.toString();
        }
    }

    native void drawFrame(int shaderProgram);
    static {
        System.loadLibrary("simplendk");
    }
    
}
